<?php
// This file is part of Moodle - http://moodle.org/
//
// Moodle is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
//
// Moodle is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with Moodle.  If not, see <http://www.gnu.org/licenses/>.

/**
 * lib file
 *
 * @package    local_kopere_mobile
 * @copyright  2024 Eduardo Kraus {@link http://eduardokraus.com}
 * @license    http://www.gnu.org/copyleft/gpl.html GNU GPL v3 or later
 */

// phpcs:disable

/**
 * @param stored_file $storedfile
 * @param null $lifetime
 * @param int $filter
 * @param array $options
 *
 * @throws coding_exception
 */
function localscorm_send_stored_file($storedfile, $lifetime, $options) {
    global $CFG;

    if (empty($options['filename'])) {
        $filename = null;
    } else {
        $filename = $options['filename'];
    }

    if ($lifetime === 'default' || is_null($lifetime)) {
        $lifetime = $CFG->filelifetime;
    }

    if (!empty($options['preview'])) {
        // Replace the file with its preview
        $fs = get_file_storage();
        $previewfile = $fs->get_file_preview($storedfile, $options['preview']);
        if (!$previewfile) {
            // Unable to create a preview of the file, send its default mime icon instead
            if ($options['preview'] === 'tinyicon') {
                $size = 24;
            } else if ($options['preview'] === 'thumb') {
                $size = 90;
            } else {
                $size = 256;
            }
            $fileicon = file_file_icon($storedfile, $size);
            localscorm_send_file($CFG->dirroot . '/pix/' . $fileicon . '.png', basename($fileicon) . '.png');
        } else {
            // Preview images have fixed cache lifetime and they ignore forced download
            // (they are generated by GD and therefore they are considered reasonably safe).
            $storedfile = $previewfile;
            $lifetime = DAYSECS;
        }
    }

    // Handle external resource
    if ($storedfile && $storedfile->is_external_file() && !isset($options['sendcachedexternalfile'])) {
        $storedfile->send_file($lifetime, 0, false, $options);
        die;
    }

    if (!$storedfile || $storedfile->is_directory()) {
        // Nothing to serve
        header("die: 05");
        die;
    }

    $filename = is_null($filename) ? $storedfile->get_filename() : $filename;

    // Use given MIME type if specified.
    $mimetype = $storedfile->get_mimetype();

    // Allow cross-origin requests only for Web Services.
    // This allow to receive requests done by Web Workers or webapps in different domains.
    if (WS_SERVER) {
        header('Access-Control-Allow-Origin: *');
    }

    localscorm_send_file($storedfile, $filename, $lifetime, 0, false, $mimetype, $options);
}

/**
 * @param        $path
 * @param        $filename
 * @param null $lifetime
 * @param int $filter
 * @param bool $pathisstring
 * @param string $mimetype
 * @param array $options
 *
 * @throws coding_exception
 */
function localscorm_send_file($path, $filename, $lifetime = null, $filter = 0, $pathisstring = false, $mimetype = '', array $options = []) {
    global $CFG, $COURSE;

    if ($lifetime === 'default' || is_null($lifetime)) {
        $lifetime = $CFG->filelifetime;
    }

    if (is_object($path)) {
        $pathisstring = false;
    }

    \core\session\manager::write_close(); // Unlock session during file serving.

    // Use given MIME type if specified, otherwise guess it.
    if (!$mimetype || $mimetype === 'document/unknown') {
        $mimetype = get_mimetype_for_sending($filename);
    }

    // If user is using IE, urlencode the filename so that multibyte file name will show up correctly on popup
    if (core_useragent::is_ie() || core_useragent::is_edge()) {
        $filename = rawurlencode($filename);
    }

    header('Content-Disposition: inline; filename="' . $filename . '"');

    if ($lifetime > 0) {
        $immutable = '';
        if (!empty($options['immutable'])) {
            $immutable = ', immutable';
            // Overwrite lifetime accordingly:
            // 90 days only - based on Moodle point release cadence being every 3 months.
            $lifetimemin = 60 * 60 * 24 * 90;
            $lifetime = max($lifetime, $lifetimemin);
        }
        $cacheability = ' public,';
        if (!empty($options['cacheability']) && ($options['cacheability'] === 'public')) {
            // This file must be cache-able by both browsers and proxies.
            $cacheability = ' public,';
        } else if (!empty($options['cacheability']) && ($options['cacheability'] === 'private')) {
            // This file must be cache-able only by browsers.
            $cacheability = ' private,';
        } else if (isloggedin() && !isguestuser()) {
            // By default, under the conditions above, this file must be cache-able only by browsers.
            $cacheability = ' private,';
        }
        $nobyteserving = false;
        header('Cache-Control:' . $cacheability . ' max-age=' . $lifetime . ', no-transform' . $immutable);
        header('Expires: ' . gmdate('D, d M Y H:i:s', time() + $lifetime) . ' GMT');
        header('Pragma: ');

    } else { // Do not cache files in proxies and browsers
        $nobyteserving = true;
        if (is_https()) { // HTTPS sites - watch out for IE! KB812935 and KB316431.
            header('Cache-Control: private, max-age=10, no-transform');
            header('Expires: ' . gmdate('D, d M Y H:i:s', 0) . ' GMT');
            header('Pragma: ');
        } else { //normal http - prevent caching at all cost
            header('Cache-Control: private, must-revalidate, pre-check=0, post-check=0, max-age=0, no-transform');
            header('Expires: ' . gmdate('D, d M Y H:i:s', 0) . ' GMT');
            header('Pragma: no-cache');
        }
    }

    if (empty($filter)) {
        // Send the contents
        if ($pathisstring) {
            readstring_accel($path, $mimetype);
        } else {
            localscorm_readfile_accel($path, $mimetype);
        }

    } else {
        // Try to put the file through filters
        if ($mimetype == 'text/html' || $mimetype == 'application/xhtml+xml') {
            $options = new stdClass();
            $options->noclean = true;
            $options->nocache = true; // Temporary workaround for MDL-5136
            if (is_object($path)) {
                $text = $path->get_content();
            } else if ($pathisstring) {
                $text = $path;
            } else {
                $text = implode('', file($path));
            }
            $output = format_text($text, FORMAT_HTML, $options, $COURSE->id);

            readstring_accel($output, $mimetype);

        } else if (($mimetype == 'text/plain') && ($filter == 1)) {
            // Only filter text if filter all files is selected
            $options = new stdClass();
            $options->newlines = false;
            $options->noclean = true;
            if (is_object($path)) {
                $text = htmlentities($path->get_content(), ENT_QUOTES, 'UTF-8');
            } else if ($pathisstring) {
                $text = htmlentities($path, ENT_QUOTES, 'UTF-8');
            } else {
                $text = htmlentities(implode('', file($path)), ENT_QUOTES, 'UTF-8');
            }
            $output = '<pre>' . format_text($text, FORMAT_MOODLE, $options, $COURSE->id) . '</pre>';

            readstring_accel($output, $mimetype);

        } else {
            // Send the contents
            if ($pathisstring) {
                readstring_accel($path, $mimetype);
            } else {
                localscorm_readfile_accel($path, $mimetype);
            }
        }
    }
    //die; //no more chars to output!!!
}

/**
 * @param $file
 * @param $mimetype
 */
function localscorm_readfile_accel($file, $mimetype) {
    if ($mimetype === 'text/plain') {
        // There is no encoding specified in text files, we need something consistent
        header('Content-Type: text/plain; charset=utf-8');
    } else {
        header('Content-Type: ' . $mimetype);
    }

    $lastmodified = is_object($file) ? $file->get_timemodified() : filemtime($file);
    header('Last-Modified: ' . gmdate('D, d M Y H:i:s', $lastmodified) . ' GMT');

    if (is_object($file)) {
        header('Etag: "' . $file->get_contenthash() . '"');
        if (isset($_SERVER['HTTP_IF_NONE_MATCH']) && trim($_SERVER['HTTP_IF_NONE_MATCH'], '"') === $file->get_contenthash()) {
            header('HTTP/1.1 304 Not Modified');
            return;
        }
    }

    // If etag present for stored file rely on it exclusively
    if (!empty($_SERVER['HTTP_IF_MODIFIED_SINCE']) && (empty($_SERVER['HTTP_IF_NONE_MATCH']) || !is_object($file))) {
        // Get unixtime of request header; clip extra junk off first
        $since = strtotime(preg_replace('/;.*$/', '', $_SERVER["HTTP_IF_MODIFIED_SINCE"]));
        if ($since && $since >= $lastmodified) {
            header('HTTP/1.1 304 Not Modified');
            return;
        }
    }

    header('Accept-Ranges: none');

    $filesize = is_object($file) ? $file->get_filesize() : filesize($file);

    header('Last-Modified: ' . gmdate('D, d M Y H:i:s', $lastmodified) . ' GMT');

    if ($filesize > 10000000) {
        // For large files try to flush and close all buffers to conserve memory
        while (@ob_get_level()) {
            if (!@ob_end_flush()) {
                break;
            }
        }
    }

    // Send the whole file content
    if (is_object($file)) {
        $file->readfile();
    } else {
        readfile_allow_large($file, $filesize);
    }
}